/** @file   particle.h
 * @brief   Declaration of Particle - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_PARTICLE_H
#define H_ENG2D_PARTICLE_H

#include "eng2d_dll.h"
#include "interactive.h"
#include "Gradient.h"
#include "vec2d.h"


namespace eng2d {

/** @class  Particle
 * @brief   Represents a single particle.
 * @author  Tomi Lamminsaari
 *
 * Particles are simple graphical objects that can move and change their color.
 * When several individual particles are used together, we can use them to
 * create blood stains, sparks or fragments of a breaking glass.
 *
 * The attributes are public so that you can access them directly.
 *
 * This class provides very simple operations and attributes the particle can
 * have. If you need more complex particles you can derive your own particle
 * glass from this and still make it work with the particle systems declared
 * in this library.
 *
 * The color is defined by providing a Gradient for the particle. The actual
 * color the particle gets depends on the gradient and current lifetime. When
 * the particle is in the beginning of its life circle we use the ending colors
 * of the gradient. As the lifetime of the particle closes to the zero, we
 * use the colors closer to the beginning color of the gradient.
 */
class DLLIMPORT Particle : public Interactive
{
public:

  ///
  /// Members
  /// =======

  /** Current position of this particle. */
  Vec2D m_pos;

  /** How long this particle still lives (number of updates)*/
  int m_timeleft;

  /** The original maximum lifetime. */
  int m_maxlife;

  /** Color of this particle */
  Gradient m_grad;

  /** Direction this particle is moving. */
  Vec2D m_dirVec;

  /** Size of the particle. Supported sizes are 1 and 2. */
  int m_size;

  /** Friction. On each update we multiply the @c m_dirVec by this value. */
  float m_friction;

  /** Tells if this Particle has died. This member gets valid value after
   * you've called the @c update() - method.
   */
  bool m_dead;


  ///
  /// Constants, datatype and static members
  /// ======================================


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	Particle();

	/** Destructor
   */
	virtual ~Particle();
  

private:
  Particle( const Particle& rO );
  Particle& operator = ( const Particle& rO );
public:


  ///
  /// Methods
  /// =======

  /** Updates this particle.
   * @return    <code>true</code> if this particle is dead.
   */
  virtual void update();
  

  /** Redraws this particle.
   * @param     pB                Pointer to target bitmap
   * @param     offset            Needed when translating the particle
   *                              coordinate to the bitmap-coordinate.
   */
  virtual void redraw( BITMAP* pB, const Vec2D& offset ) const;
  

  /** Sets the color of this Particle. The particle will have this same
   * color during its lifetime.
   * @param     rC                The color
   */
  virtual void setColor( const Color& rC );
  
  /** Sets the lifetime of us.
   * @param     ltime             How many update-rounds we're going to live.
   */
  virtual void lifetime( int ltime );
};

};  // end of namespace

#endif
